<!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Odtwarzacz MP3 i Radio RMF FM</title>
    <style>
        :root {
            --primary-color: #90EE90;
            --background-color: #000000;
            --panel-color: #333333;
            --border-color: #444444;
            --highlight-color: #555555;
            --text-color: #90EE90;
            --progress-background: #222222;
        }

        * {
            box-sizing: border-box;
        }

        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 2vh;
            background-color: var(--background-color);
            color: var(--text-color);
            display: flex;
            flex-direction: column;
            height: 100vh;
            user-select: none;
            touch-action: manipulation;
            overflow: hidden;
        }

        h1 {
            color: var(--primary-color);
            margin: 0 0 2vh 0;
            border-bottom: 1px solid var(--border-color);
            padding-bottom: 1.5vh;
            font-size: clamp(18px, 4vw, 28px);
        }

        .container {
            display: flex;
            flex: 1;
            gap: 2vh;
            min-height: 0;
            flex-direction: column;
        }

        @media (min-width: 768px) {
            .container {
                flex-direction: row;
            }
        }

        .explorer {
            flex: 1;
            background-color: var(--panel-color);
            border-radius: 8px;
            padding: 2vh;
            overflow-y: auto;
            box-shadow: 0 2px 5px rgba(0,0,0,0.5);
            border: 1px solid var(--border-color);
            min-height: 30vh;
        }

        .player {
            flex: 2;
            background-color: var(--panel-color);
            border-radius: 8px;
            padding: 2vh;
            box-shadow: 0 2px 5px rgba(0,0,0,0.5);
            display: flex;
            flex-direction: column;
            border: 1px solid var(--border-color);
            min-height: 60vh;
        }

        .controls {
            display: flex;
            gap: 1vh;
            margin-bottom: 2vh;
            flex-wrap: wrap;
            align-items: center;
            justify-content: center;
        }

        button {
            padding: 1.5vh;
            cursor: pointer;
            background-color: var(--highlight-color);
            color: var(--primary-color);
            border: none;
            border-radius: 4px;
            transition: background-color 0.2s;
            font-size: clamp(12px, 3vw, 16px);
            width: 8vh;
            height: 8vh;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }

        @media (min-width: 768px) {
            button {
                width: auto;
                height: auto;
                padding: 1vh 2vh;
            }
        }

        button:hover {
            background-color: #666666;
        }

        button:active {
            background-color: #777777;
        }

        .volume-control {
            display: flex;
            align-items: center;
            gap: 1vh;
            width: 100%;
            justify-content: center;
            flex-shrink: 0;
        }

        @media (min-width: 768px) {
            .volume-control {
                width: auto;
                justify-content: flex-start;
            }
        }

        .progress-container {
            margin: 2vh 0;
        }

        progress {
            width: 100%;
            height: 1vh;
            border-radius: 3px;
            background-color: var(--progress-background);
        }

        progress::-webkit-progress-bar {
            background-color: var(--progress-background);
        }

        progress::-webkit-progress-value {
            background-color: var(--primary-color);
        }

        progress::-moz-progress-bar {
            background-color: var(--primary-color);
        }

        .time-display {
            display: flex;
            justify-content: space-between;
            font-size: clamp(10px, 2.5vw, 14px);
            color: var(--text-color);
            margin-top: 0.5vh;
        }

        #fileList {
            list-style-type: none;
            padding: 0;
            margin: 0;
        }

        #fileList li {
            padding: 1.5vh 2vh;
            border-bottom: 1px solid var(--border-color);
            cursor: pointer;
            display: flex;
            align-items: center;
            color: var(--text-color);
            font-size: clamp(12px, 3vw, 16px);
        }

        #fileList li:hover {
            background-color: var(--highlight-color);
        }

        #fileList li.folder {
            font-weight: bold;
        }

        #fileList li.file {
            padding-left: 4vh;
        }

        #fileList li.playing {
            background-color: var(--highlight-color);
            color: var(--primary-color);
        }

        #nowPlaying {
            margin: 2vh 0;
            padding: 1.5vh;
            background-color: var(--highlight-color);
            border-radius: 4px;
            font-weight: bold;
            border-left: 4px solid var(--primary-color);
            color: var(--primary-color);
            font-size: clamp(14px, 3.5vw, 18px);
            text-align: center;
            word-break: break-word;
        }

        .file-icon {
            margin-right: 1.5vh;
            font-size: clamp(14px, 3.5vw, 18px);
        }

        .player-content {
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        .radio-frame {
            flex: 1;
            margin-top: 2vh;
            background-color: var(--highlight-color);
            border-radius: 4px;
            border: 1px solid var(--border-color);
            overflow: hidden;
            min-height: 30vh;
        }

        .radio-frame iframe {
            width: 100%;
            height: 100%;
            border: none;
        }

        input[type="range"] {
            -webkit-appearance: none;
            height: 1vh;
            background: var(--progress-background);
            border-radius: 3px;
            width: 20vh;
            flex-shrink: 0;
        }

        input[type="range"]::-webkit-slider-thumb {
            -webkit-appearance: none;
            width: 3vh;
            height: 3vh;
            background: var(--primary-color);
            border-radius: 50%;
            cursor: pointer;
        }
    </style>
</head>
<body oncontextmenu="return false;">
    <h1>Odtwarzacz MP3 i Radio RMF FM</h1>
    
    <div class="container">
        <div class="explorer">
            <ul id="fileList"></ul>
        </div>
        
        <div class="player">
            <div class="player-content">
                <audio id="audioPlayer"></audio>
                
                <div id="nowPlaying">Wybierz utwór z listy</div>
                
                <div class="progress-container">
                    <progress id="progressBar" value="0" max="100"></progress>
                    <div class="time-display">
                        <span id="currentTime">0:00</span>
                        <span id="duration">0:00</span>
                    </div>
                </div>
                
                <div class="controls">
                    <button id="prevBtn" title="Poprzedni">◄◄</button>
                    <button id="playBtn" title="Odtwarzaj">►</button>
                    <button id="pauseBtn" title="Pauza">❚❚</button>
                    <button id="stopBtn" title="Stop">■</button>
                    <button id="nextBtn" title="Następny">►►</button>
                    
                    <div class="volume-control">
                        <span>🔊</span>
                        <input type="range" id="volumeControl" min="0" max="1" step="0.1" value="0.7" title="Głośność">
                    </div>
                </div>
                
                <div class="radio-frame">
                    <iframe src="https://www.rmfon.pl/stacja/rmf-fm" title="Strona RMF FM"></iframe>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Blokada prawidłowego przycisku myszy i dostępu do kodu
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
        
        document.addEventListener('keydown', function(e) {
            // Blokada F12, Ctrl+Shift+I, Ctrl+Shift+J, Ctrl+U
            if (e.key === 'F12' || 
                (e.ctrlKey && e.shiftKey && e.key === 'I') || 
                (e.ctrlKey && e.shiftKey && e.key === 'J') || 
                (e.ctrlKey && e.key === 'u')) {
                e.preventDefault();
            }
        });

        document.addEventListener('DOMContentLoaded', function() {
            const audioPlayer = document.getElementById('audioPlayer');
            const playBtn = document.getElementById('playBtn');
            const pauseBtn = document.getElementById('pauseBtn');
            const stopBtn = document.getElementById('stopBtn');
            const prevBtn = document.getElementById('prevBtn');
            const nextBtn = document.getElementById('nextBtn');
            const fileList = document.getElementById('fileList');
            const nowPlaying = document.getElementById('nowPlaying');
            const progressBar = document.getElementById('progressBar');
            const currentTimeDisplay = document.getElementById('currentTime');
            const durationDisplay = document.getElementById('duration');
            const volumeControl = document.getElementById('volumeControl');
            
            const baseDir = 'mp3';
            let currentPath = '';
            let songs = [];
            let currentSongIndex = -1;
            let isPaused = false;
            let pausePosition = 0;
            
            // Initialize volume
            audioPlayer.volume = volumeControl.value;
            
            // Load initial directory
            loadDirectory('');
            
            // File list navigation
            fileList.addEventListener('click', function(e) {
                const li = e.target.closest('li');
                if (!li) return;
                
                const path = li.getAttribute('data-path');
                const type = li.classList.contains('folder') ? 'folder' : 'file';
                
                if (type === 'folder') {
                    loadDirectory(path);
                } else {
                    playSelectedFile(path, li.textContent.replace('📁 ', '').replace('🎵 ', ''));
                }
            });
            
            // Play button
            playBtn.addEventListener('click', function() {
                if (isPaused) {
                    // Resume from paused position
                    audioPlayer.currentTime = pausePosition;
                    audioPlayer.play();
                    isPaused = false;
                    nowPlaying.textContent = nowPlaying.textContent.replace('Pauza: ', '');
                } else if (currentSongIndex === -1 && songs.length > 0) {
                    // Start playing if nothing is playing
                    currentSongIndex = 0;
                    playSong();
                } else if (currentSongIndex !== -1) {
                    // Play current song
                    audioPlayer.play();
                    updateNowPlaying();
                }
            });
            
            // Pause button
            pauseBtn.addEventListener('click', function() {
                if (!audioPlayer.paused) {
                    pausePosition = audioPlayer.currentTime;
                    audioPlayer.pause();
                    isPaused = true;
                    nowPlaying.textContent = 'Pauza: ' + nowPlaying.textContent.replace('Pauza: ', '');
                }
            });
            
            // Stop button
            stopBtn.addEventListener('click', function() {
                audioPlayer.pause();
                audioPlayer.currentTime = 0;
                pausePosition = 0;
                isPaused = false;
                nowPlaying.textContent = 'Odtwarzanie zatrzymane';
                
                // Remove playing highlight
                Array.from(fileList.children).forEach(li => {
                    li.classList.remove('playing');
                });
            });
            
            // Previous button
            prevBtn.addEventListener('click', function() {
                if (songs.length === 0) return;
                
                if (currentSongIndex <= 0) {
                    currentSongIndex = songs.length - 1;
                } else {
                    currentSongIndex--;
                }
                
                isPaused = false;
                playSong();
            });
            
            // Next button
            nextBtn.addEventListener('click', function() {
                if (songs.length === 0) return;
                
                if (currentSongIndex >= songs.length - 1) {
                    currentSongIndex = 0;
                } else {
                    currentSongIndex++;
                }
                
                isPaused = false;
                playSong();
            });
            
            // Volume control
            volumeControl.addEventListener('input', function() {
                audioPlayer.volume = this.value;
            });
            
            // Progress bar
            audioPlayer.addEventListener('timeupdate', function() {
                if (audioPlayer.duration) {
                    const progress = (audioPlayer.currentTime / audioPlayer.duration) * 100;
                    progressBar.value = progress;
                    
                    currentTimeDisplay.textContent = formatTime(audioPlayer.currentTime);
                }
            });
            
            // Click on progress bar to seek
            progressBar.addEventListener('click', function(e) {
                if (audioPlayer.duration) {
                    const percent = e.offsetX / this.offsetWidth;
                    audioPlayer.currentTime = percent * audioPlayer.duration;
                    if (isPaused) {
                        pausePosition = audioPlayer.currentTime;
                    }
                }
            });
            
            // Update duration when metadata is loaded
            audioPlayer.addEventListener('loadedmetadata', function() {
                durationDisplay.textContent = formatTime(audioPlayer.duration);
            });
            
            // Auto play next song when current ends
            audioPlayer.addEventListener('ended', function() {
                if (songs.length > 0) {
                    if (currentSongIndex >= songs.length - 1) {
                        currentSongIndex = 0;
                    } else {
                        currentSongIndex++;
                    }
                    isPaused = false;
                    playSong();
                }
            });
            
            function loadDirectory(path) {
                currentPath = path;
                
                fetch('get_directory.php?path=' + encodeURIComponent(path))
                    .then(response => response.json())
                    .then(data => {
                        renderFileList(data);
                        // Update songs list with all files in current directory
                        songs = data.files.map(file => currentPath ? currentPath + '/' + file : file);
                    })
                    .catch(error => console.error('Error:', error));
            }
            
            function renderFileList(files) {
                fileList.innerHTML = '';
                
                // Add parent directory link (except for root)
                if (currentPath !== '') {
                    const parentLi = document.createElement('li');
                    parentLi.className = 'folder';
                    parentLi.setAttribute('data-path', currentPath.split('/').slice(0, -1).join('/'));
                    
                    const icon = document.createElement('span');
                    icon.className = 'file-icon';
                    icon.textContent = '📁';
                    
                    parentLi.appendChild(icon);
                    parentLi.appendChild(document.createTextNode('..'));
                    fileList.appendChild(parentLi);
                }
                
                // Add folders
                files.folders.forEach(folder => {
                    const li = document.createElement('li');
                    li.className = 'folder';
                    li.setAttribute('data-path', currentPath ? currentPath + '/' + folder : folder);
                    
                    const icon = document.createElement('span');
                    icon.className = 'file-icon';
                    icon.textContent = '📁';
                    
                    li.appendChild(icon);
                    li.appendChild(document.createTextNode(folder));
                    fileList.appendChild(li);
                });
                
                // Add files
                files.files.forEach(file => {
                    const li = document.createElement('li');
                    li.className = 'file';
                    li.setAttribute('data-path', currentPath ? currentPath + '/' + file : file);
                    
                    const icon = document.createElement('span');
                    icon.className = 'file-icon';
                    icon.textContent = '🎵';
                    
                    li.appendChild(icon);
                    li.appendChild(document.createTextNode(file.replace(/^.*[\\\/]/, '')));
                    fileList.appendChild(li);
                });
            }
            
            function playSelectedFile(path, name) {
                // Update songs list with all files in current directory
                songs = Array.from(fileList.querySelectorAll('li.file')).map(li => li.getAttribute('data-path'));
                currentSongIndex = songs.indexOf(path);
                
                // Play the selected file
                isPaused = false;
                playSong();
                nowPlaying.textContent = 'Teraz odtwarzane: ' + name;
            }
            
            function playSong() {
                if (currentSongIndex === -1 || songs.length === 0) return;
                
                const songPath = songs[currentSongIndex];
                audioPlayer.src = baseDir + '/' + songPath;
                audioPlayer.play();
                
                // Highlight current song in file list
                Array.from(fileList.children).forEach(li => {
                    li.classList.remove('playing');
                    if (li.getAttribute('data-path') === songPath) {
                        li.classList.add('playing');
                        li.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                    }
                });
                
                // Update now playing text
                const songName = songPath.split('/').pop();
                nowPlaying.textContent = 'Teraz odtwarzane: ' + songName;
            }
            
            function updateNowPlaying() {
                if (currentSongIndex !== -1 && songs.length > 0) {
                    const songName = songs[currentSongIndex].split('/').pop();
                    nowPlaying.textContent = 'Teraz odtwarzane: ' + songName;
                }
            }
            
            function formatTime(seconds) {
                const minutes = Math.floor(seconds / 60);
                const secs = Math.floor(seconds % 60);
                return `${minutes}:${secs < 10 ? '0' : ''}${secs}`;
            }
        });
    </script>
</body>
</html>